using System;
using System.IO;
using System.Xml;
using System.Reflection;
using System.Collections;
using System.Runtime.Remoting;

namespace Team_Project
{
	/// <summary>
	/// Motore di Team Project. Si occupa dell'inizializzazione del sistema ed
	///  l'anello principale della catena. Il suo compito  quello di smistare
	/// i messaggi alle unit funzionali adibite.
	/// </summary>
	public class TPEngine:MarshalByRefObject, IOperationHandler, IDisposable
	{
		private Hashtable handlers = new Hashtable();
		private ArrayList ToDispose = new ArrayList();
		/// <summary>
		/// Token che identifica l'oggetto all'interno della catena
		/// </summary>
		protected object chTkn;
		private ObjRef globs;
		/// <summary>
		/// Costruisce un'istanza del motore
		/// </summary>
		public TPEngine()
		{
			string fileName = @"..\..\..\Config\Engine.xml";
			string asmName, typeName;
			Assembly asm;
			Type t;

			Globals.Instance.Engine = this;

			XmlDocument doc = new XmlDocument();
			doc.Load(fileName);

			#region Init Types
			foreach(XmlNode nd in doc.SelectNodes("//Component"))
			{
				asmName = nd.Attributes["Assembly"].Value;
				asm = Assembly.Load(asmName);
				typeName = nd.Attributes["Type"].Value;
				t = asm.GetType(typeName);
				MethodInfo mi = t.GetMethod("ComponentInit",new Type[0]);
				object ino = mi.Invoke(null,null);
				TeamProjectInitAttribute tpca = (TeamProjectInitAttribute)
					t.GetCustomAttributes(typeof(TeamProjectInitAttribute),false)[0];
				if(tpca.ToDispose)
					ToDispose.Add(ino);
				if(tpca.PutInGlobal)
					Globals.Instance.Data[tpca.Name] = ino;
			}
			#endregion

			#region Load Modules
			foreach(XmlNode nd in doc.SelectNodes("//Module"))
			{
				asmName = nd.Attributes["Assembly"].Value;
				asm = Assembly.Load(asmName);
				typeName = nd.Attributes["Type"].Value;
				t = asm.GetType(typeName);
				ITPModule mod = (ITPModule) Activator.CreateInstance(t);
				mod.RegisterOperations(this);
			}
			#endregion

			globs = RemotingServices.Marshal(Globals.Instance,"Globals.rem");
		}

		#region IOperationHandler Members
		/// <summary>
		/// Imposta o restituisce il token che identifica l'engine all'interno della
		/// catena
		/// </summary>
		public object ChainToken
		{
			get{return chTkn;}
			set{chTkn = value;}
		}

		/// <summary>
		/// Vedi <see cref="Team_Project.IOperationHandler"/>
		/// </summary>
		/// <param name="data"></param>
		/// <param name="fullXml"></param>
		/// <param name="ok"></param>
		/// <returns></returns>
		public string Compute(IDictionary data,string fullXml, out bool ok)
		{
			XmlTextReader xtr = new XmlTextReader(new StringReader(fullXml));
			Stream result;
			xtr.WhitespaceHandling = WhitespaceHandling.None;
			if(!xtr.Read() || xtr.Name != "TeamProjectRequest")
			{
				ok = false;
				result = FormatError("TeamProjectRequest");
			}
			else
				result = Compute(data,xtr,out ok);
			StreamReader sr = new StreamReader(result);
			return sr.ReadToEnd();
		}
		/// <summary>
		/// Vedi <see cref="Team_Project.IOperationHandler"/>
		/// </summary>
		/// <param name="data"></param>
		/// <param name="tr"></param>
		/// <param name="ok"></param>
		/// <returns></returns>
		public System.IO.Stream Compute(IDictionary data, System.Xml.XmlTextReader tr,out bool ok)
		{
			ok = false;
			if(!tr.Read() || tr.Name != "Project")
				return FormatError("Project");
			if(!tr.Read() || tr.Name != "Name")
				return FormatError("Name");
			if(!tr.Read() || tr.NodeType != XmlNodeType.Text)
				return FormatError("Name text");
			string prjName = tr.Value;
			if(prjName == string.Empty)
				return FormatError("Project Name not empty");
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return FormatError("Name end element");
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return FormatError("Project end element");

			if(!tr.Read() || tr.Name != "Action")
				return FormatError("Action");
			if(!tr.Read() || tr.Name != "Code")
				return FormatError("Action Code");
			if(!tr.Read() || tr.NodeType != XmlNodeType.Text)
				return FormatError("Action Code text");
			string action = tr.Value;
			if(prjName == string.Empty)
				return FormatError("Action Code not empty");
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return FormatError("Action Code end element");
			
			data["project"] = prjName;
			data["operation"]= action;
			IOperationHandler hdl = (IOperationHandler) handlers[action];
			if(hdl == null)
				return CustomError("Unknown Action");
			bool ok2;
			Stream res = hdl.Compute(data,tr,out ok2);
			if(!ok2) return res;

			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return FormatError("Action end element");

			ok = true;
			return res;
		}

		#endregion

		/// <summary>
		/// Registra un handler di operazione per le richieste.
		/// </summary>
		/// <param name="commandString">Nome del comando</param>
		/// <param name="hndl">Istanza dell'handler</param>
		public void Register(string commandString, IOperationHandler hndl)
		{
			handlers[commandString] = hndl;
		}

		/// <summary>
		/// Deregistra un handler di operazione per le richieste.
		/// </summary>
		/// <param name="commandString">Nome del comando</param>
		public void UnRegister(string commandString)
		{
			handlers.Remove(commandString);
		}

		/// <summary>
		/// Scrive l'xml di risposta relativo ad un errore di formato
		/// nella richiesta
		/// </summary>
		/// <example>
		/// <code>
		/// if(!xtr.Read() || xtr.Name != "TeamProjectRequest")
		/// {
		///		ok = false;
		///		result = FormatError("TeamProjectRequest");
		///	}
		///	</code></example>
		/// <param name="expected">Descrizione dell'elemento richiesto</param>
		/// <returns>Un memory stream contenente la stringa xml di errore</returns>
		public static Stream FormatError(string expected)
		{
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("Error");
			tw.WriteString("Request format error: expected " + expected);
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			return ms;
		}
		/// <summary>
		/// Scrive l'xml di risposta relativo ad un errore.
		/// </summary>
		/// <param name="message">Messaggio di errore</param>
		/// <returns>Un memory stream contenente la stringa xml di errore</returns>
		public static Stream CustomError(string message)
		{
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("Error");
			tw.WriteString(message);
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			return ms;
		}
		
		/// <summary>
		/// Scrive l'xml di risposta relativo ad un errore.
		/// </summary>
		/// <param name="excTime">Descrizione dell'operazione che si stava
		/// eseguendo quando  stata generata l'eccezione</param>
		/// <param name="e">Eccezione che  stata generata</param>
		/// <returns>Un memory stream contenente la stringa xml di errore</returns>
		public static Stream ExceptionError(string excTime, Exception e)
		{
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("Error");
			tw.WriteString("An error has occured while " + excTime + ": " + e.Message);
			System.Diagnostics.Trace.WriteLine("An error has occured while " + excTime + ": " + e.StackTrace);
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			return ms;
		}

		#region IDisposable Members
		/// <summary>
		/// Rilascia tutte le risorse occupate e chiama il metodo dispose su
		/// tutti gli oggetti inizializzati all'avvio che ne hanno fatto richiesta
		/// </summary>
		public void Dispose()
		{
			foreach(IDisposable o in ToDispose)
			{
				o.Dispose();
			}
			RemotingServices.Unmarshal(globs);
		}

		#endregion
	}
}
